<?php

require_once(dirname(__FILE__) ."/lib.php");

class local_passerelle_inscriptions extends local_passerelle_external_api {

    //////////////////////////////////////////////////////////////////
    // cad_etudiant_inscrire
    public static function etudiant_inscrire_parameters() {
        return new external_function_parameters(
            array(
                'inscription' => new external_single_structure(
                    array(
                        'codepermanent' => new external_value(PARAM_ALPHANUM, "Code permanent de l'étudiant en minuscules : abcd12345678"),
                        'inc_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba (inc_seq_no)"),
                        'cours'         => new external_value(PARAM_ALPHANUMEXT, "Cours en majuscules sans tiraits : 123123FD"),
                        'option'        => new external_value(PARAM_ALPHANUM, "Option de cours : 60"),
                        'version'       => new external_value(PARAM_ALPHANUM, "Version du cours sur deux chiffres : 01"),
                        'crs_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel du cours dans Coba (inc_crs_seq_no)"),
                        'annee'         => new external_value(PARAM_INT, "Année de l'inscription : YYYY"),
                        'session'       => new external_value(PARAM_ALPHANUM, "Session de l'inscription : H, E ou A"),
                        'programme'     => new external_value(PARAM_ALPHANUM, "Programme auquel l'inscription est liée, en majuscules  (inc_pgm): 300A0, 08001, 41012, JEE0K, etc."),
                        'annee_version_pgm' => new external_value(PARAM_ALPHANUMEXT, "Année-version-programme de l'inscription : YYYY"),
                        'date_inscription' => new external_value(PARAM_ALPHANUMEXT, "Date d'inscription au cours : YYYY-MM-DD"),
                        'groupe'        => new external_value(PARAM_INT, "Numéro du groupe auquel l'étudiant est inscrit : 000-999"),
                        'date_expedition' => new external_value(PARAM_ALPHANUMEXT, "Date d'expédition des documents : YYYY-MM-DD"),
                        'date_confirmation' => new external_value(PARAM_ALPHANUMEXT, "Date de confirmation : YYYY-MM-DD"),
                        'date_echeance_devoir' => new external_value(PARAM_ALPHANUMEXT, "Date d'échéance des devoirs : YYYY-MM-DD"),
                        'serie_devoir'  => new external_value(PARAM_ALPHA, "Série de devoir de l'étudiant : A-W"),
                    )
                )
            )
        );
    }
    public static function etudiant_inscrire_returns() {
        return new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Moodle");
    }
    /**
     * 
     * 
     * @global type $DB
     * @global type $CFG
     * @param type $inscription
     * @return int Numéro d'inscription Moodle
     * @throws ws_exception
     */
    public static function etudiant_inscrire($inscription) {
        global $DB, $CFG;
        
        require_once($CFG->libdir .'/enrollib.php');
        require_once($CFG->dirroot .'/group/lib.php');

        $debug_info = array();

        // Check 1 serie_devoir: S'assurer que la valeur de la serie est valide
        if(empty($inscription['serie_devoir']))
             throw new ws_exception("manque_paramètre", "Le parametre serie_devoir est obligatoire");

        /*
        // TODO : donne une erreur de passage par référence. Voir si c'est nécessaire 
         * de faire l'appel validate_parameters()
        $inscription = array_pop(self::validate_parameters(
                self::etudiant_inscrire_parameters(), 
                array('inscription' => $inscription_usager)));
        */
  
        // Instancier le plugin d'inscription
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('etudiant');

        // Instancier le cours
        $course = local_passerelle_inscriptions::_get_course($inscription['cours'], $inscription['option'], $inscription['version']);
        array_push($debug_info, 'courseid:'.$course->id);

        // Check 2 serie_devoir:  S'assurer que la serie existe 
        require_once($CFG->libdir .'/gradelib.php');
        $item = grade_item::fetch(array('courseid'=>$course->id,'idnumber'=>'D1'.$inscription['serie_devoir']));
        if(!$item)
             throw new ws_exception("serie_devoir_invalide", "Oups, la serie_devoir est invalide!");
        
        $context = context_course::instance($course->id, IGNORE_MISSING);
        self::validate_context($context);
        require_capability('enrol/etudiant:enrol', $context);

        ////
        // Vérifier l'existence du rôle étudiant et la possibilité de l'assigner
        $roles = get_assignable_roles($context);

        if (!array_key_exists(local_passerelle_external_api::ROLE_ETUDIANT, $roles)) {
            throw new ws_exception("manque_de_droit", "L'utilisateur auquel le token d'authentification est lié ne dispose pas des droits suffisants pour assigner le rôle d'étudiant.");
        }

        ////
        // Trouver l'instance du plugin d'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($course->id, true);

       
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'etudiant') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        // Le plugin d'inscription n'a pas été trouvé, c'est donc la première insription d'un étudiant au cours
        // il faut activer le plugin pour ce cours en créant une instance.
        if (empty($enrolinstance)) {
            $enrolid = $enrol->add_instance($course);
            $enrolinstance = $DB->get_record('enrol', array('id'=>$enrolid));
            array_push($debug_info, "Ajout de l'instance $enrolid au cours {$course->id}.");
        }

        ////
        // Trouver l'etudiant
        $student = $DB->get_record('user', array('username'=>strtolower($inscription['codepermanent'])));
        if (!$student) {
            // on ne doit pas employer invalid_parmeter_exception, car on ne pourra pas modifier le message qui est prédéfini
            // throw new invalid_parameter_exception("L'usager demandé ({$inscription['codepermanent']}) n'existe pas dans Moodle.");
            throw new ws_exception("erreur_usager_inexistant", "L'usager demandé ({$inscription['codepermanent']}) n'existe pas dans Moodle.");
        }

        ////
        // Trouver le groupe
        $num_cours = self::_get_course_option_version($inscription['cours'], $inscription['option'], $inscription['version']);
        $group = groups_get_group_by_idnumber($course->id, $num_cours .'_'. $inscription['groupe']);
        if (!$group) {
            throw new ws_exception("erreur_groupe_inexistant", "Le groupe demandé ({$inscription['groupe']}) n'existe pas.");
        }

        ////
        // Est-ce qu'il y a déjà une inscription active de cet étudiant à ce cours?
        $ue = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$student->id));
        if ($ue) {
            // si deja inscrit, actif et dans le meme groupe, on retourne l'id de l'inscription
            $groups = groups_get_user_groups($course->id, $student->id);

            $inscrit_au_groupe = in_array($group->id, $groups[0]);
            if (($ue->status == 0) && $inscrit_au_groupe) {
                // TODO: comparer le inc_seq_no pour être bien certain, vraiment, qu'on parle de la même inscription.
                return (int) $ue->id;
            }
            // sinon, erreur.
            $err_groupe = 'groupe '. ($inscrit_au_groupe ? 'identique' : 'différent');
            $err_statut = 'statut '. ($ue->status ? 'inactif' : 'actif'); // pourquoi est-ce que 0 veut dire actif dans Moodle?
            throw new ws_exception("erreur_etudiant_deja_inscrit", "L'étudiant est déjà inscrit à ce cours : user_enrolment.id : ".
                "{$ue->id}, {$err_statut}, {$err_groupe}.");
        }

        ////
        // Démarrer la transaction, pour pouvoir faire un rollback en cas d'erreur...
        $transaction = $DB->start_delegated_transaction();

        // Vérifier que les données sont cohérentes
        $ue_incseqno = $DB->get_record('enrol_etudiant', array('inc_seq_no' => $inscription['inc_seq_no']));
        if ($ue_incseqno) {
            if (($ue_incseqno->userid != $student->id)) {
                throw new ws_exception("inc_seq_no_deja_utilise", "Le inc_seq_no {$inscription->inc_seq_no} est déjà utilisé (id : {$ue_incseqno->id}).");
            }
        }

        ////
        // Sauvegarder les données spécifiques de l'inscription

        // C'est tentant d'utiliser le inc_seq_no pour la requête qui suit, mais 
        // ça ne détecterait pas le cas où un étudiant fait une reprise du cours.
        $ue_current_data = $DB->get_record('enrol_etudiant', array('userid'=>$student->id, 'courseid'=>$course->id));
        if ($ue_current_data) {
            // l'étudiant a déjà des données d'inscription - on réutilise la ligne.
            // TODO: quand la procédure de désinscription sera faite, déterminer si c'est bien que les lignes 
            // restent après la désinscription (oui c'est bien) et selons quels critères ont les efface après un certain temps (à voir)
            $data = clone $ue_current_data;
            $debug_info[] = $data;
        } else {
            $data = new stdClass();
        }


        //print_R($inscription);
        $date_inscription   = DateTime::createFromFormat('Y-m-d', $inscription['date_inscription']);
        $date_expedition    = DateTime::createFromFormat('Y-m-d', $inscription['date_expedition']);
        $date_confirmation  = DateTime::createFromFormat('Y-m-d', $inscription['date_confirmation']);
        // TODO: ajuster la date de fin d'inscription
        $date_echeance = DateTime::createFromFormat('Y-m-d', $inscription['date_echeance_devoir']);

        $err_date = false;
        $str_date_err = array();
        foreach(array('date_inscription', 'date_expedition', 'date_echeance', 'date_confirmation') as $str_date) {
            if (!$$str_date) {
                $err_date = true;
                $str_date_err[] = $str_date;
            }
        }
        if ($err_date) {
            throw new ws_exception("erreur_date_manquante", "Les paramètres de date sont obligatoire. Champs manquants : ". join(', ', $str_date_err) .")");
        }
        
        $data->inc_seq_no = $inscription['inc_seq_no'];
        $data->userid = $student->id;
        $data->courseid = $course->id;
        $data->annee = $inscription['annee'];
        $data->session = $inscription['session'];
        $data->programme = $inscription['programme'];
        $data->annee_version_pgm = $inscription['annee_version_pgm'];
        $data->date_inscription = $date_inscription->getTimestamp();
        $data->date_expedition = $date_expedition->getTimestamp();
        $data->date_confirmation = $date_confirmation->getTimestamp();
        $data->groupe = $inscription['groupe'];
        $data->date_echeance_devoir = $date_echeance->getTimestamp();
        $data->serie_devoir = $inscription['serie_devoir'];
        $data->serie_examen = $inscription['serie_examen'];

        if (!property_exists($data, 'id')) {
            $data_id = $DB->insert_record('enrol_etudiant', $data);
            if (!$data_id) {
                throw new ws_exception("erreur_base_de_donnees", "Une erreur est survenue lors de la sauvegarde des données d'inscription de l'étudiant.");
            }
        } else {
            $DB->update_record('enrol_etudiant', $data);
        }

        //// 
        // Inscrire l'utilisateur au cours
        $enrol->enrol_user($enrolinstance, $student->id, local_passerelle_external_api::ROLE_ETUDIANT, $date_inscription->getTimestamp());/*, $date_echeance->getTimestamp()*/

        //// 
        // Ajouter l'étudiant au groupe
        $group_added = groups_add_member($group, $student);
        if (!$group_added) {
            throw new ws_exception("erreur_base_de_donnees", "L'ajout de l'étudiant ({$inscription['codepermanent']}) au groupe demandé ({$inscription['groupe']}) a échoué.");
        }
        
        ////
        // Tout s'est bien passé, faire le commit de la transaction
        $transaction->allow_commit();

        if (!empty($debug_info)) {
            // var_dump($debug_info);
        }
        
        // trouver le numéro de l'inscription pour la valeur de retour
        $eid = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$student->id));
        return (int) $eid->id;
    }
 
    //////////////////////////////////////////////////////////////////
    // cad_etudiant_terminer_cours
    public static function etudiant_terminer_cours_parameters() {

          return new external_function_parameters(
            array(
                 'inc_seq_no' => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba : inc_seq_no")
            )
        );
    }
    public static function etudiant_terminer_cours_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen.");
    }
    public static function etudiant_terminer_cours($inc_seq_no) {
        global $DB, $CFG,$PAGE;
        require_once($CFG->libdir .'/enrollib.php');

        $debug_info = array();
        // Termine
        // Retrouver l'inscription de l'étudiant
        $ue_current_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=>$inc_seq_no));

        if ($ue_current_data) {
            $data = clone $ue_current_data;
            $debug_info[] = $data;
        } else {
           // throw new ws_exception("erreur_etudiant_non_inscrit", "Le numéro d'inscription fourni est introuvable dans la base de données.");
           throw new ws_exception("erreur_etudiant_incseqno_inexistant", "Le numéro d'inscription inc_seq_no ($inc_seq_no) est introuvable dans la BD.");
        }

        // Instancier le plugin etudiant
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('etudiant');
        
        // Récupérer l'instance de l'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($data->courseid,true);

      
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'etudiant') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        
        if (empty($enrolinstance)) {
            throw new ws_exception("erreur_plugin_manquant", "Houston, we have a problem. Le plugin d'inscription etudiant a été enlevé du cours, mais l'étudiant y est inscrit. Ça ne devrait pas arriver. Il faut réactiver le plugin avant de pouvoir continuer.");
        }
        
        if($enrol->unenrol_user($enrolinstance,$data->userid))
            return true;

        return false;
    }
    
    
    //////////////////////////////////////////////////////////////////
    // cad_tuteur_inscrire
    public static function tuteur_inscrire_parameters() {
        return new external_function_parameters(
            array(
                'inscription' => new external_single_structure(
                    array(
                        'codetuteur'    => new external_value(PARAM_ALPHANUM, "Numéro de tuteur (username), nombre entie sans zéro initial"),
                        'cours'         => new external_value(PARAM_ALPHANUMEXT, "Cours en majuscules sans tiraits : 123123FD"),
                        'option'        => new external_value(PARAM_ALPHANUM, "Option de cours : 60"),
                        'version'       => new external_value(PARAM_ALPHANUM, "Version du cours sur deux chiffres : 01"),
                        'crs_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel du cours dans Coba (inc_crs_seq_no)"),
                        'date_debut'    => new external_value(PARAM_ALPHANUMEXT, "Date de début du tuteur pour ce cours : YYYY-MM-DD"),
                        'groupe'        => new external_value(PARAM_INT, "Numéro du groupe auquel le tuteur est assigné : 000-999"),
                        'temporaire'    => new external_value(PARAM_INT, "Indicateur de remplacement temporaire : 0 (non), 1 (oui)"),
                    )
                )
            )
        );
    }
    public static function tuteur_inscrire_returns() {
        return new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Moodle");
    }
    /**
     * @global type $DB
     * @global type $CFG
     * @param type $inscription
     * @return int Numéro d'inscription Moodle
     * @throws ws_exception
     */
    public static function tuteur_inscrire($inscription) {
        global $DB, $CFG;

        require_once($CFG->libdir .'/enrollib.php');
        require_once($CFG->dirroot .'/group/lib.php');

        $debug_info = array();

        ////
        // Instancier le plugin d'inscription
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('tuteur');
        $course = local_passerelle_inscriptions::_get_course($inscription['cours'], $inscription['option'], $inscription['version']);
        array_push($debug_info, 'courseid:'.$course->id);

        $context = context_course::instance($course->id, IGNORE_MISSING);
        self::validate_context($context);        
        require_capability('enrol/tuteur:enrol', $context);

        ////
        // Vérifier l'existence du rôle tuteur et la possibilité de l'assigner
        $roles = get_assignable_roles($context);

        if (!array_key_exists(local_passerelle_external_api::ROLE_TUTEUR, $roles)) {
            throw new ws_exception("erreur_permission_insuffisante", "Permissions insuffisantes pour assigner le rôle de tuteur.");
        }
  
        ////
        // Trouver l'instance du plugin d'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($course->id, true);
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'tuteur') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        // Le plugin d'inscription n'a pas été trouvé, c'est donc la première insription d'un tuteur au cours
        // il faut activer le plugin pour ce cours en créant une instance.
        if (empty($enrolinstance)) {
            $enrolid = $enrol->add_instance($course);
            $enrolinstance = $DB->get_record('enrol', array('id'=>$enrolid));
            array_push($debug_info, "Ajout de l'instance $enrolid au cours {$course->id}.");
        }

        ////
        // Trouver le tuteur
        $tuteur = $DB->get_record('user', array('username'=>$inscription['codetuteur']));
        if (!$tuteur) {
            throw new ws_exception("erreur_usager_inexistant", "L'usager {$inscription['codetuteur']} n'existe pas dans Moodle.");
        }

        ////
        // Trouver le groupe
        $num_cours = self::_get_course_option_version($inscription['cours'], $inscription['option'], $inscription['version']);
        $group = groups_get_group_by_idnumber($course->id, $num_cours .'_'. $inscription['groupe']);
        if (!$group) {
            throw new ws_exception("erreur_groupe_inexistant", "Le groupe {$inscription['groupe']} n'existe pas.");
        }

        ////
        // Est-ce qu'il y a déjà une inscription active de ce tuteur à ce cours?
        $tuteur_already_enrolled = false;
        $ue = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$tuteur->id));
        if ($ue) {
            // Le tuteur est déjà inscrit
            $tuteur_already_enrolled = true;
            $actif = ($ue->status ? 'inactif' : 'actif');

            // Réactiver l'inscription si elle ne l'est pas déjà
            if (!$actif) {

                $ue->status = 0;
                $DB->update_record('user_enrolments', $ue);
            }

          //  return (int) $ue->id;
        }
        
        ////
        // Démarrer la transaction, pour pouvoir faire un rollback en cas d'erreur...
        $transaction = $DB->start_delegated_transaction();

        ////
        // Sauvegarder les données spécifiques de l'inscription

        // C'est tentant d'utiliser le inc_seq_no pour la requête qui suit, mais 
        // ça ne détecterait pas le cas où un étudiant fait une reprise du cours.
        $ue_current_data = $DB->get_record('enrol_tuteur', array('userid'=>$tuteur->id, 'courseid'=>$course->id,'groupe'=>$inscription['groupe']));
        if ($ue_current_data) {
            // l'étudiant a déjà des données d'inscription - on réutilise la ligne.
            // TODO: quand la procédure de désinscription sera faite, déterminer si c'est bien que les lignes 
            // restent après la désinscription (oui c'est bien) et selons quels critères ont les efface après un certain temps (à voir)
            $data = clone $ue_current_data;
            $debug_info[] = $data;
        } else {
            $data = new stdClass();
        }


        $timestart = DateTime::createFromFormat('Y-m-d', $inscription['date_debut']);

        // TODO: ajuster la date de fin d'inscription
        // $timeend = DateTime::createFromFormat('Y-m-d', $inscription['date_echeance_devoir']);

        $data->crs_seq_no = $inscription['crs_seq_no'];
        $data->userid = $tuteur->id;
        $data->courseid = $course->id;
        $data->date_inscription = $timestart->getTimestamp();

        $data->groupe = $inscription['groupe'];
        $data->temporaire = $inscription['temporaire'];

        if (!$data->id) { 
            $data_id = $DB->insert_record('enrol_tuteur', $data);
            if (!$data_id) {
                throw new ws_exception("erreur_base_de_donnees", "Une erreur est survenue lors de la sauvegarde des données d'inscription du tuteur.");
            }
        }
        else {
            $DB->update_record('enrol_tuteur', $data);
        }

        //// 
        // Inscrire le tuteur au cours, s'il n'y est pas déjà
        if (!$tuteur_already_enrolled) {
            $enrol->enrol_user($enrolinstance, $tuteur->id, local_passerelle_external_api::ROLE_TUTEUR, $timestart->getTimestamp());
        }

        //// 
        // Ajouter le tuteur au groupe
        $group_added = groups_add_member($group, $tuteur);
        if (!$group_added) {
            throw new ws_exception("erreur_base_de_donnees", "L'ajout du tuteur {$inscription['codetuteur']} au groupe {$inscription['groupe']} a échoué.");
        }

        ////
        // Tout s'est bien passé, faire le commit de la transaction
        $transaction->allow_commit();

        if (!empty($debug_info)) {
            // var_dump($debug_info);
        }
        
        // trouver le numéro de l'inscription pour la valeur de retour
        $eid = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$tuteur->id));
        return (int) $eid->id;
    }

    //////////////////////////////////////////////////////////////////
    // cad_tuteur_desinscrire
    public static function tuteur_desinscrire_parameters() {
        return new external_function_parameters(
            array(
                'inscription' => new external_single_structure(
                    array(
                        'codetuteur' => new external_value(PARAM_ALPHANUM, "Numéro de tuteur (nombre entier), sans zéro initial"),
                        'cours'         => new external_value(PARAM_ALPHANUMEXT, "Cours en majuscules sans tiraits : 123123FD"),
                        'option'        => new external_value(PARAM_ALPHANUM, "Option de cours : 60"),
                        'version'       => new external_value(PARAM_ALPHANUM, "Version du cours sur deux chiffres : 01"),
                        'crs_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel du cours dans Coba (inc_crs_seq_no)"),
                    )
                )
            )
        );
    }
    public static function tuteur_desinscrire_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen");
    }
    public static function tuteur_desinscrire($inscription) {
        global $DB, $CFG;

        require_once($CFG->libdir .'/enrollib.php');
        
        return false; // TODO : le code qui suit est une copie de etudiant_desinscrire

        $debug_info = array();

        // Instancier le plugin d'inscription
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('tuteur');        
                
        return false; // TODO : le code qui suit est une copie de etudiant_desinscrire

        $debug_info = array();

        $course = local_passerelle_inscriptions::_get_course($inscription['cours'], $inscription['option'], $inscription['version']);
        $context = context_course::instance($course->id, IGNORE_MISSING);
        self::validate_context($context);        
        require_capability('enrol/etudiant:unenrol', $context);

        ////
        // Trouver l'instance du plugin d'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($course->id, true);
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'etudiant') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        if (empty($enrolinstance)) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "Il n'y a pas d'inscriptions d'étudiants dans le cours {$cours->id}.");
        }

        ////
        // Trouver l'etudiant
        $student = $DB->get_record('user', array('idnumber'=>$inscription['codepermanent']));
        if (!$student) {
            throw new ws_exception("erreur_usager_inexistant", "L'usager demandé ({$inscription['codepermanent']}) n'existe pas dans Moodle.");
        }

        ////
        // Trouver l'inscription à désactiver
        $ue = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$student->id));
        if (!$ue) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "L'étudiant {$inscription['codepermanent']} n'est pas inscrit au cours {$cours->id}.");
        }

        $transaction = $DB->start_delegated_transaction();

        ////
        // Suspendre ou effacer l'inscription
        // TODO: déterminer le traitement à faire, la durée où l'inscription est inactive avant d'être effacée, etc.
        $enrol->update_user_enrol($enrolinstance, $student->id, 1); // 0 = actif, 1 = inactif. 

        ////
        // TODO : déterminer à quel moment on supprime complètement l'inscription, les notes de l'étudiant, etc.
        // Faire ça ici.

        ////
        // Tout s'est bien passé, on faire le commit de la transaction
        $transaction->allow_commit();

        return true;
    }

    private static function _get_enrol_plugin($type) {
        $enrol = enrol_get_plugin($type);
        if (empty($enrol)) {
            throw new ws_exception("erreur_plugin_manquant", "Le plugin d'inscription '$type' n'est pas activé.");
        }
        return $enrol;
    }

    
    private static function _get_course($cours, $option, $version) {
        global $DB;
        
        // TODO: confirmer qu'on utilise cours-option-version comme idnumber et pas crs_seq_no!
        $num_cours = self::_get_course_option_version($cours, $option, $version);
        
        $course = $DB->get_record('course', array('idnumber'=>$num_cours));
        if (!$course) {
            throw new ws_exception("erreur_cours_inexistant", "Le cours demandé ($num_cours) n'est pas présent dans Moodle");
        }
        
        return $course;
    }

    private static function _get_course_option_version($cours, $option, $version){
        return strtoupper("$cours-$option-$version");
    }

    private static function _get_option_from_shortname($shortname) {
        $cours = explode('-', $shortname);
        return $cours[3];
    }

    
    /**
     * Task #626
     * @return external_function_parameters
     */
    //////////////////////////////////////////////////////////////////
    // cad_etudiant_inactiver_cours
    public static function etudiant_inactiver_cours_parameters() {
        return new external_function_parameters(
            array(
                'inc_seq_no' => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba (inc_seq_no)")
                )
        );
    }
    public static function etudiant_inactiver_cours_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen.");
    }
    public static function etudiant_inactiver_cours($inc_seq_no) {
        global $DB, $CFG;

        require_once($CFG->libdir .'/enrollib.php');

        $debug_info = array();

        ////
        // Instancier le plugin d'inscription
        $enrol = enrol_get_plugin('etudiant');
        if (empty($enrol)) {
            throw new ws_exception("erreur_plugin_manquant", "Le plugin d'inscription des étudiants n'est pas activé.");
        }

        ///
        //Trouver le record d'enrol
        $ue_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=> $inc_seq_no));
        if (!$ue_data) {
            throw new ws_exception("erreur_etudiant_incseqno_inexistant", "Le inc_seq_no demandé ($inc_seq_no) n'est pas inscrit dans Moodle");
        }

        $context = context_course::instance($ue_data->courseid, IGNORE_MISSING);
        self::validate_context($context);
        require_capability('enrol/etudiant:unenrol', $context);

        ////
        // Trouver l'instance du plugin d'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($ue_data->courseid, true);
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'etudiant') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        if (empty($enrolinstance)) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "Il n'y a pas d'inscriptions d'étudiants dans le cours {$ue_data->courseid}.");
        }

        ////
        // Trouver l'inscription à désactiver
        $ue = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$ue_data->userid));
        if (!$ue) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "L'étudiant n'est pas inscrit au cours {$ue_data->courseid}.");
        }

        $transaction = $DB->start_delegated_transaction();

        ////
        // Suspendre ou effacer l'inscription
        $enrol->update_user_enrol($enrolinstance, $ue_data->userid, 1); // 0 = actif, 1 = inactif.

        ////
        // Tout s'est bien passé, on faire le commit de la transaction
        $transaction->allow_commit();

        return true;
    }

    /**
     * Task #626
     * @return external_function_parameters
     */
    //////////////////////////////////////////////////////////////////
    // cad_etudiant_reactiver_debloquer
    public static function etudiant_reactiver_cours_parameters() {
        return new external_function_parameters(
            array(
                'inc_seq_no' => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba (inc_seq_no)")
            )
        );
    }
    public static function etudiant_reactiver_cours_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen.");
    }
    public static function etudiant_reactiver_cours($inc_seq_no) {
        global $DB, $CFG;

        require_once($CFG->libdir .'/enrollib.php');

        $debug_info = array();

        ////
        // Instancier le plugin d'inscription
        $enrol = enrol_get_plugin('etudiant');
        if (empty($enrol)) {
            throw new ws_exception("erreur_plugin_manquant", "Le plugin d'inscription des étudiants n'est pas activé.");
        }

        ///
        //Trouver le record d'enrol
        $ue_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=> $inc_seq_no));
        if (!$ue_data) {
            throw new ws_exception("erreur_etudiant_incseqno_inexistant", "Le inc_seq_no demandé ($inc_seq_no) n'est pas inscrit dans Moodle");
        }

        $context = context_course::instance($ue_data->courseid, IGNORE_MISSING);
        self::validate_context($context);
        require_capability('enrol/etudiant:unenrol', $context);

        ////
        // Trouver l'instance du plugin d'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($ue_data->courseid, true);
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'etudiant') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        if (empty($enrolinstance)) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "Il n'y a pas d'inscriptions d'étudiants dans le cours {$ue_data->courseid}.");
        }

        ////
        // Trouver l'inscription à désactiver
        $ue = $DB->get_record('user_enrolments', array('enrolid'=>$enrolinstance->id, 'userid'=>$ue_data->userid));
        if (!$ue) {
            throw new ws_exception("erreur_etudiant_non_inscrit", "L'étudiant n'est pas inscrit au cours {$ue_data->courseid}.");
        }

        $transaction = $DB->start_delegated_transaction();

        ////
        // Suspendre ou effacer l'inscription
        $enrol->update_user_enrol($enrolinstance, $ue_data->userid, 0); // 0 = actif, 1 = inactif.

        ////
        // Tout s'est bien passé, on faire le commit de la transaction
        $transaction->allow_commit();

        return true;
    }



    //////////////////////////////////////////////////////////////////
    // cad_etudiant_prolonger_cours
    public static function etudiant_prolonger_cours_parameters() {
       
           return new external_function_parameters(
                array(
                    'prolongation' => new external_single_structure(
                        array(
                            'inc_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba (inc_seq_no)"),
                            'date_echeance' => new external_value(PARAM_ALPHANUMEXT, "Date d'échéance: YYYY-MM-DD (inc_dt_echeance_devoir || inc_dt_expiration)"),
                            'type'          => new external_value(PARAM_INT,"Type de la date d'échéance 1 ou 2 ( 1: devoir / 2: examen)")
                        )
                    )
                )   
            );
    }

  
    public static function etudiant_prolonger_cours_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen");
    }
    public static function etudiant_prolonger_cours($prolongation) {
        global $DB, $CFG;

        require_once($CFG->libdir .'/enrollib.php');

        // Vérification
        // 1 - Inscription existante
        $ue_current_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=>$prolongation['inc_seq_no']));
        if(!$ue_current_data)
            // throw new ws_exception("erreur_etudiant_non_inscrit_err1", "inc_seq_no non trouvé.");
            throw new ws_exception("erreur_etudiant_incseqno_inexistant", "inc_seq_no ({$prolongation['inc_seq_no']}) non trouvé.");

        $data = clone $ue_current_data;
        $debug_info[] = $data;

        // Traitement
        $date_echeance      = strtotime($prolongation['date_echeance']); //DateTime::createFromFormat('Y-m-d', $prolongation['date_echeance']);
        $date_echeance_type = $prolongation['type'];       //cleaned

        // l'étudiant a déjà des données d'inscription - on réutilise la ligne.
        $enrolinstance = reset($DB->get_records('enrol', array('courseid'=>$ue_current_data->courseid,'enrol'=>'etudiant')));
        if(is_object($enrolinstance))
        {
            $current_user_enrolinstance = $DB->get_record('user_enrolments', array('userid'=>$ue_current_data->userid,'enrolid'=>$enrolinstance->id));    
            $current_course_instance    = $DB->get_record('course', array('id'=>$ue_current_data->courseid));
            switch($date_echeance_type){
                case 1: //update
                    $data->date_echeance_devoir = $date_echeance;
                    $DB->update_record('enrol_etudiant', $data); 
                    $string_prolongation_evaluation_type = "la date d'échéance du devoir"; 
                break;
                case 2: // Update    
                    $data->date_echeance_examen = $date_echeance;
                    $DB->update_record('enrol_etudiant', $data);
                    $string_prolongation_evaluation_type = "la date d'échéance de l'évaluation finale";  
                break;
                default: 
                    //@ return error code
                    throw new ws_exception("erreur_date_echeance_type", "Le type de la date d'écheance n'est pas reconnu.");
            } 

                // 
                // Création de l'evenement
                require_once($CFG->dirroot .'/calendar/lib.php');
                $event = new stdClass;
                $event->name         = "Prolongation de ".$string_prolongation_evaluation_type;
                $event->description  = "Prolongation de ".$string_prolongation_evaluation_type." ".$current_course_instance->fullname;
                $event->courseid     = 0;
                $event->groupid      = 0;
                $event->userid       = $ue_current_data->userid;//$current_user->id;
                
                $event->timestart    = $date_echeance;
                $event->timeduration = 0;
                $event->visible      = true;
                $event->uuid         = $prolongation['inc_seq_no'];

                $transaction    = $DB->start_delegated_transaction();
                $event_result   = calendar_event::create($event);
                $transaction    = $transaction->allow_commit();

        }
        else
            throw new ws_exception("erreur_etudiant_non_inscrit_err2", "enrol_instance non trouvé.");
   
        return true;
    }



    //////////////////////////////////////////////////////////////////
    // cad_etudiant_ouvrir_examen
    public static function etudiant_ouvrir_examen_parameters() {
        return new external_function_parameters(
            array(
                'examen_serie' => new external_single_structure(
                    array(
                        'inc_seq_no'    => new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Coba (inc_seq_no)"),
                        'serie_examen'  => new external_value(PARAM_ALPHA, "Série d'examen de l'étudiant : A-W (INC_SERIE_EXAMEN)"),
			'date_echeance_examen' => new external_value(PARAM_ALPHANUMEXT, "Date d'échéance des examens : YYYY-MM-DD (inc_dt_expiration)"),
            'facteur_mesure_examen' => new external_value(PARAM_INT, "Facteur de mesure de l'étudiant pour l'évaluation (facteur_mesure_examen)", VALUE_OPTIONAL),
            'ind_reprise' => new external_value(PARAM_INT, "indicateur de reprise de l'examen (ind_reprise)", VALUE_OPTIONAL)
                    )
                )
            )
        );
    }
    public static function etudiant_ouvrir_examen_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen.");
    }
    public static function etudiant_ouvrir_examen($examen_serie) {
        global $DB, $CFG;
        require_once($CFG->libdir .'/enrollib.php');
        /* Ajouter date echeance examen dans enrol_etudiant */
        // Vérifier si l'etudiant existe
        $ue_current_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=>$examen_serie['inc_seq_no']));
        if(!$ue_current_data)
           //  throw new ws_exception("erreur_etudiant_non_inscrit_err1", "inc_seq_no non trouvé.");
           throw new ws_exception("erreur_etudiant_incseqno_inexistant", "inc_seq_no ({$examen_serie['inc_seq_no']}) non trouvé.");
     
        // l'étudiant a déjà des données d'inscription - on réutilise la ligne.
        // TODO: quand la procédure de désinscription sera faite, déterminer si c'est bien que les lignes 
        // restent après la désinscription (oui c'est bien) et selons quels critères ont les efface après un certain temps (à voir)
        $data = clone $ue_current_data;
        $debug_info[] = $data;
     
        // !important : Vérifier si la serie existe

        $date_echeance_examen = DateTime::createFromFormat('Y-m-d', $examen_serie['date_echeance_examen']);
        
        $enrolinstance = reset($DB->get_records('enrol', array('courseid'=>$ue_current_data->courseid,'enrol'=>'etudiant')));
        if(!is_object($enrolinstance))
            throw new ws_exception("erreur_etudiant_non_inscrit_err2", "enrol_instance non trouvé.");
        
        $data->serie_examen = $examen_serie['serie_examen'];
        $data->date_echeance_examen = $date_echeance_examen->getTimestamp();
        $data->facteur_mesure_examen = (isset($examen_serie['facteur_mesure_examen'])) ? $examen_serie['facteur_mesure_examen'] : 0;

        $DB->update_record('enrol_etudiant', $data);
        
        return true;
    }


    //////////////////////////////////////////////////////////////////
    // cad_tuteur_remplacer
    public static function tuteur_remplacer_parameters() {
        return new external_function_parameters(
            array(
                'inscription' => new external_single_structure(
                    array(
                        'codetuteur'    => new external_value(PARAM_ALPHANUM, "Numéro de tuteur (nombre entier), sans zéro initial"),
                        'cours'         => new external_value(PARAM_ALPHANUMEXT, "Cours en majuscules sans tiraits : 123123FD"),
                        'option'        => new external_value(PARAM_ALPHANUM, "Option de cours : 60"),
                        'version'       => new external_value(PARAM_ALPHANUM, "Version du cours sur deux chiffres : 01"),
                        'groupe'        => new external_value(PARAM_INT, "Numéro du groupe auquel le tuteur est assigné : 000-999")
                    )
                )
            )
        );
    }
    public static function tuteur_remplacer_returns() {
        return new external_value(PARAM_BOOL, "Succès de l'opération : booléen");
    }
    public static function tuteur_remplacer($inscription) {
        global $DB, $CFG,$PAGE;

        // 1- Trouver le tuteur courrant
        // 1-1 Vérifier si le cours existe
        $num_cours = self::_get_course_option_version($inscription['cours'],$inscription['option'], $inscription['version']);
        $course = $DB->get_record('course', array('idnumber'=>$num_cours));
        if (!$course) {
            throw new ws_exception("erreur_cours_inexistant", "Le cours demandé ($num_cours) n'est pas présent dans Moodle");
        }
        // 1-2 Vérifier si le group existe
        $group = groups_get_group_by_idnumber($course->id, $num_cours .'_'. $inscription['groupe']);
        if (!$group) {
            throw new ws_exception("erreur_groupe_inexistant", "Le groupe demandé ({$inscription['groupe']}) n'existe pas.");
        }
        // 1-3 Vérifier si le nouveau tuteur est ajouté
        $tuteur = $DB->get_record('user', array('username'=>$inscription['codetuteur']));
        if (!$tuteur) {
            throw new ws_exception("erreur_usager_inexistant", "L'usager {$inscription['codetuteur']} n'existe pas dans Moodle.");
        }

        require_once($CFG->libdir .'/enrollib.php');

        $debug_info = array();
       
        // Retrouver l'inscription courrante de l'ancien tuteur au cours-groupe
        $ue_current_data = $DB->get_record('enrol_tuteur', array('courseid'=>$course->id,'groupe'=>$inscription['groupe']));

        if ($ue_current_data) {
            $data = clone $ue_current_data;
            $debug_info[] = $data;
        } else {
            throw new ws_exception("erreur_coursgroup_sans_tuteur", "L'inscription fourni du tuteur est introuvable dans la base de données.");
        }

        // Nouveau tuteur

        // 1-4 Vérifier si le tuteur n'est pas lui même
        if($ue_current_data->userid == $tuteur->id)
            throw new ws_exception("erreur_tuteur_deja_inscrit", "Le tuteur est déjà inscrit.");
        
        // 1-5 Vérifier si le groupe contient un tuteur
        // Faire une exception si le coours-groupe n'as pas de tuteur 
        // ** Sinon si plusieurs tuteurs existes (Remplacer et désinscrire du groupes les autres tuteurs) **!
        // ** enrol_tuteur(tutorId,groupId,coursId) hase to be unique key  **!

        
        // 2- Désinscrire le tuteur courrant du cours-groupe
        // Instancier le plugin tuteur
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('tuteur');
        
        // Récupérer l'instance de l'inscription
        $enrolinstance = null;
        $enrolinstances = enrol_get_instances($data->courseid,true);

      
        foreach($enrolinstances as $courseenrolinstance) {
            if ($courseenrolinstance->enrol == 'tuteur') {
                $enrolinstance = $courseenrolinstance;
                break;
            }
        }
        
        if (empty($enrolinstance)) {
            throw new ws_exception("erreur_plugin_manquant", "Houston, we have a problem. Le plugin d'inscription etudiant a été enlevé du cours, mais l'étudiant y est inscrit. Ça ne devrait pas arriver. Il faut réactiver le plugin avant de pouvoir continuer.");
        }
        
        require_once($CFG->dirroot.'/group/lib.php');
                
        $group = groups_get_group_by_idnumber($course->id, $num_cours .'_'. $inscription['groupe']);
        if (!$group) {
            throw new ws_exception("erreur_groupe_inexistant", "Le groupe {$inscription['groupe']} n'existe pas.");
        }
        groups_remove_member($group->id, $ue_current_data->userid);
        $DB->delete_records('enrol_tuteur', array('id'=>$ue_current_data->id));
        
        $inscription['crs_seq_no'] = $ue_current_data->crs_seq_no;
        $inscription['date_debut'] = date("Y-m-d", $ue_current_data->date_inscription);
        $inscription['temporaire'] = 0;
        
        local_passerelle_inscriptions::tuteur_inscrire($inscription);

        return true;
    }

    /**
     * Task #1853
     */
    public static function etudiant_changer_group_parameters() {
        return
            new external_function_parameters(
                array(
                    'inc_seq_no' => new external_value(PARAM_TEXT, "Identifiant de l'inscription à modifier (inc_seq_no): ex 579170"),
                    'groupe' => new external_value(PARAM_TEXT, "Nouvelle valeur du groupe: ex 16")
                )
            );

    }
    public static function etudiant_changer_group_returns() {
        return new external_value(PARAM_TEXT, "Id du nouveau groupe si le groupe a été changé, ou 1 si le groupe courant était le même que le nouveau.");
    }

    public static function etudiant_changer_group($inc_seq_no, $groupe) {
        global $CFG, $DB;
        // Instancier le plugin d'inscription
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('etudiant');

        // 1. on cherche l'information de la ligne correspondante à $inc_seq_no dans la table {enrol_etudiant}
        $enrol_et = $enrol->get_user_enrolment($inc_seq_no);
        if(!$enrol_et) {
            throw new ws_exception("erreur_etudiant_incseqno_inexistant", "inc_seq_no ({$inc_seq_no}) non trouvé.");
        }
        else if(empty($enrol_et->groupe)) {
            throw new ws_exception("erreur_etudiant_incseqno_inexistant", "inc_seq_no ({$inc_seq_no}) champ groupe vide.");
        }
        else if($enrol_et->groupe == $groupe) {
            return "1";
        }

        require_once($CFG->dirroot.'/group/lib.php');

        // 2. on cherche dans la table {course} la ligne correspondante au cours auquel l'inscription se rapporte
        $course = $DB->get_record('course', array('id'=>$enrol_et->courseid));
        if (!$course) {
            throw new ws_exception("erreur_cours_inexistant", "Le cours id {$num_cours} n'est pas présent dans Campus");
        }
        else if(empty($course->idnumber)) {
            throw new ws_exception("erreur_cours_inexistant", "Le cours id {$num_cours} champ idnumber vide");
        }
        // 2.1 on a établi dans #4076-1 que le champ idnumber du cours servira pour restituer le champ idnumber de groupe
        $cours_idnumber = $course->idnumber;

        // 3. on maintenant tous les components pour vérifier la présence du groupe courante et de nouveau groupe
        // 3.1 trouver le groupe courant 
        $group_courant = groups_get_group_by_idnumber($course->id, $cours_idnumber .'_'. $enrol_et->groupe);
        if (!$group_courant) {
            throw new ws_exception("erreur_groupe_inexistant", "Groupe ({$enrol_et->groupe}) non trouvé bien que cela peu possible.");
        }
        // 3.2 trouver le groupe nouveau
        $group_nouveau = groups_get_group_by_idnumber($course->id, $cours_idnumber .'_'. $groupe);
        if (!$group_nouveau) {
            throw new ws_exception("erreur_groupe_inexistant", "Groupe ({$groupe}) n'est pas présent dans Campus.");
        }

        // 4. Appliquer la modification
        if ($group_courant && $group_nouveau) {
            groups_remove_member($group_courant, $enrol_et->userid);
            groups_add_member($group_nouveau, $enrol_et->userid);
            // TODO : si jamais un problème se manifeste considérer Delegated transactions
            if ($enrol->set_user_group($inc_seq_no, $groupe)) {
                return (int) $group_nouveau->id;
            }
        }

    }

    /**
     * Task #1854
     */
    public static function inscription_soumission_nbrdevoir_parameters() {
        return
            new external_function_parameters(
                array(
                    'inc_seq_no' => new external_value(PARAM_TEXT, "Identifiant de l'inscription à modifier (inc_seq_no): ex 579170")
                )
            );

    }
    public static function inscription_soumission_nbrdevoir_returns() {
        return new external_value(PARAM_INT, "nbr de soumissions pour cet identifiant");
    }
    public static function inscription_soumission_nbrdevoir($inc_seq_no) {

        /**
         * -- quiz query */
        global $CFG, $DB;

        // aller chercher les paramètres course et userid avec le $inc_seq_no
        $enrol = local_passerelle_inscriptions::_get_enrol_plugin('etudiant');
        $enrol_et = $enrol->get_user_enrolment($inc_seq_no);

        $course_shortname = $DB->get_field('course','shortname', array('id'=>$enrol_et->courseid));
        $option = local_passerelle_inscriptions::_get_option_from_shortname($course_shortname);

        //TODO: valider qu'il n'y a pas d'autre option papier.
        $options_papier = array(10, 14);

        if (in_array($option, $options_papier)) {
            // Pour ces cours-là, il n'y a pas de soumission. Le tuteur rentre la note
            // une fois qu'il a corrigé le devoir papier envoyé par l'élève.
            $query = " select  count(mag.id) nbr
                        from {assign_grades} mag
                        join {assign} ma on ma.id= mag.assignment
                        where ma.course = $enrol_et->courseid
                        and mag.userid = $enrol_et->userid ";

            $tot_quiz = $DB->get_field_sql($query);
            return $tot_quiz;
        }


        // chercher quizes
        $quizes = $DB->get_records('quiz', array('course'=>$enrol_et->courseid));



        // Pour simplifier j'irai avec la serie car le nom varie 'Devoir', 'Homework', 'Assignment'
        $devoirs = array();
        foreach ($quizes as $quiz) {
            $nom = trim($quiz->name);
            $nom_length = strlen($nom);
            if ($nom[$nom_length - 1] == $enrol_et->serie_devoir ){
                array_push($devoirs, $quiz->id);
            }
        }
        $tot_quiz = 0;

        if (!empty($devoirs)) { // Si c'est un cours qu'avec des assigns... pas de quiz

            $in_quizes = ' ( ' . implode(', ', $devoirs) . ')';

            // Devoirs soumis par l'étudiant
            $query = "select count(distinct(t.quizid)) nbr
                        from (
                            select qza.quiz quizid
                            FROM {quiz_attempts} qza
                            JOIN {question_attempts} qna ON qna.questionusageid = qza.uniqueid
                            JOIN {question_attempt_steps} qas ON qas.userid = qza.userid AND qas.questionattemptid = qna.id
                            JOIN (
                                SELECT userid, questionattemptid, MAX(sequencenumber) as maxseqnum
                                FROM {question_attempt_steps}
                                WHERE userid = $enrol_et->userid
                                GROUP BY questionattemptid, userid
                            ) maxsubq ON maxsubq.questionattemptid = qna.id
                            AND maxsubq.maxseqnum = qas.sequencenumber AND maxsubq.userid = qas.userid
                            LEFT JOIN {question_attempt_steps} qam ON qam.questionattemptid = qna.id
                            AND qam.state IN('mangrright',
                                'gradedright',
                                'gradedpartial',
                                'mangrpartial',
                                'gradedwrong',
                                'mangrwrong')
                            WHERE qza.quiz in $in_quizes
                            and qza.state = 'finished'
                            AND qna.behaviour = 'manualgraded'
                            AND qza.timefinish != 0) t ";

            $tot_quiz = $DB->get_field_sql($query);


        }

        $tot_assign = 0;
         $query = " select count(asgn_sub.id) asgnnbr
                    FROM {assign_submission} asgn_sub
                    JOIN {assign} a ON a.id = asgn_sub.assignment and a.course = $enrol_et->courseid
                    LEFT JOIN {assign_grades} ag ON ag.assignment = asgn_sub.assignment
                    AND ag.userid = asgn_sub.userid AND asgn_sub.attemptnumber = ag.attemptnumber
                    WHERE asgn_sub.userid = $enrol_et->userid AND asgn_sub.status = 'submitted' AND
                    a.grade <> 0";

        $tot_assign = $DB->get_field_sql($query);
        $somme = $tot_quiz + $tot_assign;


        return $somme;

    }




    //////////////////////////////////////////////////////////////////
    // cad_inscription_etudiant_modifier
    public static function inscription_etudiant_modifier_parameters() {
        return new external_function_parameters(
            array(
                    'inscription'   => new external_single_structure(
                        array(
                            'inc_seq_no'        => new external_value(PARAM_ALPHANUM, "Identificant inscription SGE"),
                            'session'           => new external_value(PARAM_ALPHANUM, "Session de l'inscription : H, E ou A",VALUE_OPTIONAL),
                            'date_confirmation' => new external_value(PARAM_ALPHANUMEXT, "Date de confirmation : YYYY-MM-DD",VALUE_OPTIONAL),
                        )
                )
            )
        );
    }
    public static function inscription_etudiant_modifier_returns() {
        return new external_value(PARAM_INT, "Numéro séquentiel de l'inscription dans Moodle");
    }
    /**
     * 
     * 
     * @global type $DB
     * @global type $CFG
     * @param type $inscription
     * @return int Numéro d'inscription Moodle
     * @throws ws_exception
     */
    public static function inscription_etudiant_modifier($inscription) {
        global $DB, $CFG,$USER, $SESSION;
        $isUpdate = false;

        // Vérification
        // 1 - Inscription existante
        $ue_current_data = $DB->get_record('enrol_etudiant', array('inc_seq_no'=>$inscription['inc_seq_no']));
        if(!$ue_current_data)
            throw new ws_exception("erreur_etudiant_non_inscrit_err1", "inc_seq_no non trouvé.");
        
        
        if(isset($inscription['session']) && in_array($inscription['session'],array('H','E','A')) )
        {
            $ue_current_data->session = $inscription['session'];
            $isUpdate = true;
        } else if (isset($inscription['session'])) {
            throw new ws_exception("Session inexistante!");
        }

        if(isset($inscription['date_confirmation']))
        {
            if (DateTime::createFromFormat('Y-m-d', $inscription['date_confirmation'])) {
                $ue_current_data->date_confirmation = DateTime::createFromFormat('Y-m-d', $inscription['date_confirmation'])->getTimestamp();
                $isUpdate = true;
            } else {
                throw new ws_exception("date_invalide", "Date invalide! Format doit être aaaa-mm-jj!");
            }
        }

        if(!$isUpdate)
            throw new ws_exception("erreur_nothing_to_update", "param non déclaré.");
        
        $DB->update_record('enrol_etudiant',$ue_current_data);

        return true;
    }
}
